/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * Copyright 1999 David Williams. All rights reserved.
 *
 * flush.c
 *
 * created: djw@djw.org, January 2, 1999.
 */
#include <stdio.h>

#include "prcp.h"

static prc_uint32_t
flush_app_info(prc_internal_t* iprc)
{
    size_t n;
    n = fwrite(iprc->header.appinfoData, 1,
               (size_t)iprc->header.appinfoSize, iprc->fp);
    if (n != (size_t)iprc->header.appinfoSize)
        return (prc_uint32_t)-1;
    return (prc_uint32_t)n;
}

static prc_uint32_t
flush_sort_info(prc_internal_t* iprc)
{
    size_t n;
    n = fwrite(iprc->header.sortinfoData, 1,
               (size_t)iprc->header.sortinfoSize, iprc->fp);
    if (n != (size_t)iprc->header.sortinfoSize)
        return (prc_uint32_t)-1;
    return (prc_uint32_t)n;
}

static int
flush_records(prc_internal_t* iprc, prc_uint32_t offset, FILE* fp)
{
    prc_internal_record_t* records = iprc->records;
    unsigned isResource = PRC_IS_RESOURCE(&iprc->header);
    prc_internal_record_t* foo;

    for (foo = records; foo != NULL; foo = foo->next) {
        union {
            prc_record_header_t   record;
            prc_resource_header_t resource;
        } uBuf;
        int rv;

        if (isResource) {
            uBuf.resource.offset = offset;
            uBuf.resource.id = foo->u_contents.resource.id;
            uBuf.resource.type = foo->u_contents.resource.type;
            rv = prcfwriteresource(fp, &uBuf.resource);
        } else {
            uBuf.record.offset = offset;
            uBuf.record.flags = foo->u_contents.record.flags;
            uBuf.record.id = foo->u_contents.record.id;
            rv = prcfwriterecord(fp, &uBuf.record);
        }

        if (rv == -1)
            return -1;
            
        offset += foo->u_contents.record.datalen;
    }
    
    fputc(0, fp);
    fputc(0, fp);

    if (iprc->header.appinfoData != NULL) {
        prc_uint32_t n = flush_app_info(iprc);

        if (n == (prc_uint32_t)-1)
            return -1;
    }

    if (iprc->header.sortinfoData != NULL) {
        prc_uint32_t n = flush_sort_info(iprc);

        if (n == (prc_uint32_t)-1)
            return -1;
    }

    for (foo = records; foo != NULL; foo = foo->next) {
        if (foo->u_contents.record.data != NULL) {
            size_t n;
            n = fwrite(foo->u_contents.record.data,
                       1,
                       (size_t)foo->u_contents.record.datalen,
                       fp);
            if (n != (size_t)foo->u_contents.record.datalen)
                return -1;
        }
    }

    return 0;
}

int
prcflush(prc_t* prc)
{
    prc_internal_t* iprc = (prc_internal_t*)prc;
    prc_uint32_t offset;
    unsigned is_resource = (prc->flags & PRC_FLAGS_RESOURCE) != 0;

    if ((iprc->mode & (PRC_OPEN_CREATE|PRC_OPEN_UPDATE|PRC_OPEN_WRITE|PRC_OPEN_TOUCH)) == 0) {
        return -1;
    }

    fseek(iprc->fp, 0L, SEEK_SET);

    if (prcfwriteheader(iprc->fp, prc) != 0)
        return -1;

    if (is_resource) {
        offset = sizeof(prc_file_t) +
            (sizeof(prc_file_resource_header_t) * prc->nrecords) +
            2;

    } else {
        offset = sizeof(prc_file_t) +
            (sizeof(prc_file_record_header_t) * prc->nrecords) +
            2;

        if (iprc->header.appinfoData != NULL)
            offset += iprc->header.appinfoSize;

        if (iprc->header.sortinfoData != NULL)
            offset += iprc->header.sortinfoSize;
    }

    return flush_records(iprc, offset, iprc->fp);
}
