/* JMC - JuggleMaster Pattern Converter
 *
 * Version 1.0
 * Copyright (C) Per Johan Persson 2001
 *
 *
 * parser.c - Parser for the JuggleMaster input file
 *
 */

#include "data.h"
#include "jmc_error.h"
#include "symtab.h"
#include "xmem.h"

int lineno = 1;
FILE* infile;
char* pptext;
char* tokbuf;

void parse_param(char* buf);
void parse_style(char* buf);
void parse_styledata(char* buf);
void parse_category(char* buf);
void parse_pattern(char* buf);

#define NUMBER_INT 0
#define NUMBER_FLOAT 1

typedef struct {
  int num_type;
  union {
    int intval;
    float floatval;
  } val;
} number;

number numbuf;
char* next_token(char* tok_dest, char* buf);
char* scan_number(number* num_dest, char* buf);
void printnum(number n);
char* strim(char* buf);

/* safe version of fgets. reallocs the buffer if it's too
   small */
char* sfgets(char* s, int size, FILE* stream) {
  int pos;
  s = fgets(s, size, stream);

  if (s == NULL)
    return NULL;

  if (s[0] == '\n')
    return s;

  if (strlen(s) == 0)
    s[0] = '\0';
  else {
    pos = strlen(s)-1;

    if (s[pos] != '\n')
      fatalerror("Internal compiler error (sfgets)\n");

    s[pos] = '\0';
  }

  return s;
}

/* main parser routine */
void parse(void) {
  char* ptr;
  int i = 0;

  lineno = 1;

  pptext = (char*)xmalloc(1024*sizeof(char));
  tokbuf = (char*)xmalloc(1024*sizeof(char));

  while ((pptext = sfgets(pptext, 1023, infile)) != NULL) {
    /* strip off leading and trailing whitespace */
    pptext = strim(pptext);

    /* last line? */
    if (strcmp(pptext, "/----------------- end ------------------") == 0)
      continue;

    /* look for errors in characters */
    ptr = pptext;
    while(*(ptr++) != '\0') {
      /* look for comments after command */
      if (i!= 0 && *ptr == ';')
	*ptr = '\0';

      /* make sure we're dealing with 7-bit ascii */
      if (!isascii(*ptr)) {
	error("Found illegal character '%c'\n", *ptr);
	continue;
      }

      i++;
    }

    /* the first character determines what type of line we're
       dealing with */
    switch (pptext[0]) {
    case '\n':
      break;
    case ';': /* empty line or comment */
      break;
    case '#': /* parameter */
      parse_param(pptext);
      break;
    case '%': /* style */
      parse_style(pptext);
      break;
    case '{': /* style data*/
      parse_styledata(pptext);
      break;
    case '/': /* category */
      parse_category(pptext);
      break;
    default:
      parse_pattern(pptext);
      break;
    }

    lineno++;
  }
}

/* parser helper functions */
/* parameter: #name=number */
void parse_param(char* buf) {
  char* ptr;

  ptr = buf+1;
  ptr = next_token(tokbuf, ptr);

  if (strlen(tokbuf) != 2) {
    error("Incorrect parameter\n");
    return;
  }

  if (!(buf[3] == '=')) {
    synerror("error in parameter\n");
    return;
  }
  ptr = &buf[4];
  
  /* next, we have a float or an integer */
  ptr = scan_number(&numbuf, ptr);

  /* check for valid parameter and set */
  if (strcmp(tokbuf, "GA") == 0 || strcmp(tokbuf, "ga") == 0) {
    if (!numbuf.num_type == NUMBER_FLOAT)
      error("Floating point number expected\n");
    else
      set_ga(numbuf.val.floatval);
  }
  else if (strcmp(tokbuf, "DR") == 0 || strcmp(tokbuf, "dr") == 0) {
    if (!numbuf.num_type == NUMBER_FLOAT)
      error("Floating point number expected\n");
    else {
      set_dr(numbuf.val.floatval);
    }
  }
  else if (strcmp(tokbuf, "HR") == 0 || strcmp(tokbuf, "hr") == 0) {
    if (!numbuf.num_type == NUMBER_FLOAT)
      error("Floating point number expected\n");
    else
      set_hr(numbuf.val.floatval);
  }
  else if (strcmp(tokbuf, "SP") == 0 || strcmp(tokbuf, "sp") == 0) {
    if (!numbuf.num_type == NUMBER_FLOAT)
      error("Floating point number expected\n");
    else
      set_sp(numbuf.val.floatval);
  }
  else if (strcmp(tokbuf, "BC") == 0 || strcmp(tokbuf, "bc") == 0) {
    if (!numbuf.num_type == NUMBER_INT)
      error("Integer expected\n");
    /* bc is ignored */
  }
  else if (strcmp(tokbuf, "BP") == 0 || strcmp(tokbuf, "bp") == 0) {
    if (!numbuf.num_type == NUMBER_INT)
      error("Integer expected\n");
    else
      set_bp(numbuf.val.intval);
  }
  else if (strcmp(tokbuf, "HD") == 0 || strcmp(tokbuf, "hd") == 0) {
    if (!numbuf.num_type == NUMBER_INT)
      error("Integer expected\n");
    else
      set_hd(numbuf.val.intval);
  }
  else if (strcmp(tokbuf, "PD") == 0 || strcmp(tokbuf, "pd") == 0) {
    if (!numbuf.num_type == NUMBER_INT)
      error("Integer expected\n");
    else
      set_pd(numbuf.val.intval);
  }
  else if (strcmp(tokbuf, "MR") == 0 || strcmp(tokbuf, "mr") == 0) {
    if (!numbuf.num_type == NUMBER_INT)
      error("Integer expected\n");
    else
      set_mr(numbuf.val.intval);
  }
  else
    error("Incorrect parameter\n");
}

/* style: % name */
void parse_style(char* buf) {
  /* the name is everything from character 1 */
  set_style(lookup_symtab(buf+1));
}

/* dirty macros! */
#define CHECK_INT if (numbuf.num_type != NUMBER_INT) {\
error("Style data must be an integer\n");\
return;}

#define CHECK_LBRACE if (strcmp(tokbuf, "{") != 0){\
synerror("'{' expected\n");\
return;}

#define CHECK_RBRACE if (strcmp(tokbuf, "}") != 0){\
synerror("'}' expected\n");\
return;}

#define CHECK_COMMA if (strcmp(tokbuf, ",") != 0){\
synerror("',' expected\n");\
return;}

/* style data: {number,number}{number,number} */
void parse_styledata(char* buf) {
  int s1, s2, s3, s4;

  char* ptr = buf+1;

  /* read number */
  ptr = scan_number(&numbuf, ptr);
  CHECK_INT;
  s1 = numbuf.val.intval;

  /* read , */
  ptr = next_token(tokbuf, ptr);
  CHECK_COMMA;

  /* read number */
  ptr = scan_number(&numbuf, ptr);
  CHECK_INT;
  s2 = numbuf.val.intval;

  /* read } */
  ptr = next_token(tokbuf, ptr);
  CHECK_RBRACE;
  
  /* read { */
  ptr = next_token(tokbuf, ptr);
  CHECK_LBRACE;
  
  /* read number */
  ptr = scan_number(&numbuf, ptr);
  CHECK_INT;
  s3 = numbuf.val.intval;

  /* read , */
  ptr = next_token(tokbuf, ptr);
  CHECK_COMMA;

  /* read number */
  ptr = scan_number(&numbuf, ptr);
  CHECK_INT;
  s4 = numbuf.val.intval;

  /* read } */
  ptr = next_token(tokbuf, ptr);
  CHECK_RBRACE;

  add_to_style(s1, s2, s3, s4);
}

/* category: /[ name ] or / name */
void parse_category(char* buf) {
  int cbracfound = 0;
  char* ptr = buf+1;

  /* check for [ */
  if (*ptr == '[') {
    while (*(ptr++) != '\0') {
      if (*ptr == ']') {
	cbracfound = 1;
	*ptr = '\0';
	break;
      }
    }
    if (!cbracfound)
      warning("missing ']' in category\n");
    ptr = buf+2;
  }

  /* everything up to nullbyte is the name */
  set_category(lookup_symtab(ptr));
}

/* pattern: site name or site */
void parse_pattern(char* buf) {
  char* ptr = buf;
  /* the site is everything up to the first
     blank or everything if no blank found */
  
  //printf("pptext[0] = '%c' %d\n", buf[0], buf[0]);

  while (*(ptr++) != '\0') {
    /* found a blank, which means that the rest
       of the line is a name */
    if (isblank(*ptr)) {
      *ptr = '\0';
      if (isblank(*(ptr+1))) ptr++;
      /* read remaining blanks */
      while (*(ptr++) != '\0')
	if (!isblank(*ptr))
	  break;

      add_pattern(buf, ptr);
      return;
    }
  }
  
  /* if we get here, no space was found, which means
     the pattern is unnamed, so we use the site for
     a name */
  add_pattern(buf, buf);
}

/* utility functions */

/* gets the next token, returns position in buffer,
   tok_dest is set to the token, tok_dest must be
   at least strlen(buf) long */
char* next_token(char* tok_dest, char* buf) {
  //fprintf(stderr, "Scanning '%s'\n", buf);

  /* skip whitespace */
  if (isblank(*buf))
    while (*(buf++) != '\0')
      if (!isblank(*buf))
	break;

  /* alphanumeric characters are appended to tok_dest */
  if (isalnum(*buf))
    while (isalnum(*buf) && *buf != '\0')
      *(tok_dest++) = *(buf++);
  /* any other character is returned */
  else
    *(tok_dest++) = *(buf++);

  *tok_dest = '\0';

  return buf;
}

/* scans for a number, returned in num_dest.
   returns the position in the buffer */
char* scan_number(number* num_dest, char* buf) {
  char nbuf[1024];
  char* pbuf = nbuf;
  int type_is_float = 0;

  /* skip whitespace */
  if (isblank(*buf))
    while (*(buf++) != '\0')
      if (!isblank(*buf))
	break;

  /* check for negative */
  if (*buf == '-')
    *(pbuf++) = *(buf++);

  /* must read at least one digit */
  if (!isdigit(*buf))
    synerror("number expected ('%c', %d)\n", *buf, *buf);

  /* scan number */
  while (*buf != '\0' && *buf != '\n') {
    if (isdigit(*buf))
      *(pbuf++) = *buf;
    else if (*buf == '.') {
      type_is_float = 1;
      *(pbuf++) = *buf;
    }
    else if (isblank(*buf))
      break;
    else {
      break;
    }

    buf++;
  }

  *pbuf = '\0';

  /* scanned a float */
  if (type_is_float) {
    num_dest->num_type = NUMBER_FLOAT;
    sscanf(nbuf, "%f", &(num_dest->val.floatval));    
  }
  /* scanned an integer */
  else {
    num_dest->num_type = NUMBER_INT;
    sscanf(nbuf, "%d", &(num_dest->val.intval));
  }

  //printnum(*num_dest);

  return buf;
}

/* trims a string of leading and trailing whitespace */
char* strim(char* buf) {
  return buf;
}

void printnum(number n) {
  if (n.num_type == NUMBER_INT)
    fprintf(stderr, "number = %d (int)\n", n.val.intval);
  else
    fprintf(stderr, "number = %f (float)\n", n.val.floatval);
}


void pperror(char* msg) {
  if (strcmp(pptext, "\n") == 0)
    error("%s\n", msg);
  else
    error("%s at '%s'\n", msg, pptext);
}

void setfile(FILE* file) {
  infile = file;
}
