/* JMC - JuggleMaster Pattern Converter
 *
 * Version 1.0
 * Copyright (C) Per Johan Persson 2001
 *
 *
 * data.h - Header file for data structures
 *
 */

/* BUGS: The pieces that are used for both JMC and JMPalm
         should be put into a separate header file
*/

#ifndef _DATA__HDR_
#define _DATA__HDR_

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#ifdef _WIN32
#include <windows.h>
#include <winsock.h>
#endif

#include "prc/prc.h"

#include "jmc_error.h"
#include "symtab.h"
#include "xmem.h"

#ifndef BOOL
#define BOOL unsigned int
#endif

#ifndef TRUE
#define TRUE (BOOL)1
#endif

#ifndef FALSE
#define FALSE (BOOL)0
#endif

#define CHECK_PARAM(param, ucase, lcase) ((strcmp((param), (ucase))==0) || (strcmp((param), (lcase))==0))

/* sets how many entries to allocate at a time */
#define MEM_INTERVAL 1024

/* same as above, but used for less heavily used
   data structures, such as categories */
#define MEM_MIN_INTERVAL 255

/* JMPalm data format structs

   Pattern database (JMPattDB.pdb)

   1       - Header data
   2       - Style data
   3       - Category definitions
   4-n     - Style definitions
   (n+1)-m - Pattern definitions
*/

#define STYLE_OFFSET 3

#define JML_MAX_NAMELEN  56
#define JML_MAX_SITELEN  56
#define JML_MAX_STYLELEN 56
#define Int8   signed char
#define UInt8  unsigned char
#define UInt16 unsigned short
#define Int16  short
#define UInt32 unsigned int
#define Int32  int

/* definitions used for database pages */
#define ID_STYLE     1 /* a style definition */
#define ID_PATTERN   2 /* a pattern definition */
#define ID_CATEGORY  3 /* a category definition */
#define ID_RECORDS   4 /* personal records */
#define ID_RAW       5 /* raw data, reserved */

extern int lineno;
void semerror(char* msg);

/* Header for the pattern library */
typedef struct {
  /* pattern and style count */
  Int16 pattern_count, style_count;
  Int32 styledata_len;              /* length of style data */
  Int16 pattern_offset, style_offset;
  UInt32 category_count;
  UInt32 record_count;
  UInt32 dbversion;   /* version of database */
  char reserved[32];  /* 32 bytes of reserved data */
} patt_header;

/* Definition of a style */
typedef struct {
  UInt8 id;                   /* id == ID_STYLE for style definitions */
  char name[JML_MAX_NAMELEN]; /* Name of the style */
  UInt16 length;              /* length of style data, (given in
				 number of lines, there are 4 bytes
				 of data for each line */
  UInt16  offset;             /* offset into style data */
} styledef;

/* Definition of a pattern */
typedef struct {
  UInt8 id;                   /* id == ID_PATTERN for pattern defintions */
  char name[JML_MAX_NAMELEN]; /* name of the pattern */
  char site[JML_MAX_SITELEN]; /* siteswap for the pattern */
  UInt32 hr, dr;              /* height ratio and dwell ratio */
  UInt32 style;               /* style definition to use, 0 for normal
				 style, record number for any other style */
  Int32 next_pattern;         /* next pattern in this category. -1 if it's
				 the last pattern in the category, 0 if it's
				 the next pattern, the record number
				 otherwise */
} patterndef;

/* Definition of a category 
   All categories are saved in one database record. The first 4
   bytes of the record stores the number of categories that are
   defined. The entire data structure is read into an array
   on program load (categories). */
typedef struct {
  char name[JML_MAX_NAMELEN];
  UInt32 first_patt;
} categorydef;

/* pointers to data */
categorydef** categories; /* array of category definitions */
patterndef**  patterns;   /* array of pattern definitions  */
styledef**    styles;     /* array of style definitions    */  
patt_header  phd;         /* the data file header          */
char*        styledata;   /* array of style data           */

/* packed size of the structs on the palm. The palm pilot
   aligns struct data on 16-bit boundaries. */

/* size of data pointers */
/* xxx_init is for the number of entries currently initialized
   xxx_size is for the number of entries currently used */
int cat_init, cat_size, patt_init, patt_size,
    style_init, style_size, styled_init, styled_size;

/* Current data */
styledef* current_style; /* pointer to current style  */
int current_style_pos;    /* position of current style */

categorydef* current_cat;

patterndef* current_patt;
int current_patt_pos;


/* Status data */

/* TRUE if the style has just been added, as opposed
   to being used for a second time. When this is true,
   a number of calls to set_style must be made next
   in order to add the actual data to the style. */
BOOL just_added; 

/* Current state */
enum state { STATE1, STATE2 };

/* Current input */
enum input { INPUT1, INPUT2 };

/* The state machine, stored as a simple matrix */
int** statemachine;

/* Initialize the data storage */
void initialize_data(void);
/* Finalize (free) the data storage */
void finalize_data(void);

/* Add methods to traverse the completed data set here */
/* should probably use STL for this */
/* FIXME: add stuff here */

/* Check the validity of the input according to the
   compiler's current state. */
//BOOL input_valid(char* err_buffer, state current_state, input current_input);

/* Change state */
//BOOL change_state(char* err_buffer, state current_state, state new_state);

/* Set a parameter */
void set_dr(float value); /* dwell ratio */
void set_hr(float value); /* height ratio */
void set_ga(float value); /* gravity (set once) */
void set_sp(float value); /* speed (set once) */
void set_bp(int value); /* beep on */
void set_hd(int value); /* hand on */
void set_pd(int value); /* pattern on */
void set_mr(int value); /* mirror */

/* Set the category for subsequent patterns added */
void set_category(char* name);
/* Set the style for subsequent patterns added */
void set_style(char* name);
/* Create a new style (called by set_style only) */
void create_style(char* name);
/* Add data to a style, this is only allowed if
   the style has just been added */
void add_to_style(int d1, int d2, int d3, int d4);
/* Add a pattern. Invalid if no category has been set.
   If no style has been set, simply use the normal style
   (which is always style 0, and never stored in the
   pattern library). */
void add_pattern(char* site, char* name);

/* Call to finalize the data, that includes:
   - update the header
   - transform relative pointers to absolute pointers */
void finalize_data(void);

void init_data(void);

/* from pdb_output.c */
void open_database(char* filename);
void write_database(void);
void close_database(void);
void new_record(void* data, unsigned short len);

/* to check endianness */
int is_big_endian(void);

/* windows specific */
#ifdef _WIN32
int isblank(int c);
#endif

/* debug */
void print_styledata(void);
void print_styledef(void);
void print_pattdef(void);
void print_cat(void);
void print_hdr(void);

#endif
